local addonName,addonTable = ...
local DA = LibStub("AceAddon-3.0"):GetAddon("Skillet") -- for DebugAids.lua
--[[
Skillet: A tradeskill window replacement.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
]]--

local isRetail = WOW_PROJECT_ID == WOW_PROJECT_MAINLINE
local isClassic = WOW_PROJECT_ID == WOW_PROJECT_CLASSIC
local isBCC = WOW_PROJECT_ID == WOW_PROJECT_BURNING_CRUSADE_CLASSIC

local PT = LibStub("LibPeriodicTable-3.1")

local skillColors = {
	["unknown"]		= { r = 1.00, g = 0.00, b = 0.00, level = 5, alttext="???", cstring = "|cffff0000"},
	["optimal"]		= { r = 1.00, g = 0.50, b = 0.25, level = 4, alttext="+++", cstring = "|cffff8040"},
	["medium"]		= { r = 1.00, g = 1.00, b = 0.00, level = 3, alttext="++",  cstring = "|cffffff00"},
	["easy"]		= { r = 0.25, g = 0.75, b = 0.25, level = 2, alttext="+",   cstring = "|cff40c000"},
	["trivial"]		= { r = 0.50, g = 0.50, b = 0.50, level = 1, alttext="",    cstring = "|cff808080"},
	["header"]		= { r = 1.00, g = 0.82, b = 0,    level = 0, alttext="",    cstring = "|cffffc800"},
}

--
-- function InitializeSkillLevels (which loads Skillet.db.global.SkillLevels) is 
-- at the end of this file.
--
-- Skillet-Classic table was scraped from https://tbc.wowhead.com/, Skillet from https://www.wowhead.com/
-- using ScrapeWowhead.js (follow instructions in file)
-- provided and maintained by Matthew Hively, https://github.com/matthewhively
--

--
-- Table of racial bonuses (including retail)
--
local racialList = {
	[4036]  = {["Gnome"] = 15,				-- engineering
			   ["KulTiran"] = 5},
	[7411]  = {["BloodElf"] = 10,			-- enchanting
			   ["KulTiran"] = 5},
	[25229] = {["Draenei"] = 5,				-- jewelcrafting
			   ["KulTiran"] = 5},
	[2259]  = {["Goblin"] = 15,				-- alchemy
			   ["KulTiran"] = 5},
	[2018]  = {["LightforgedDraenei"] = 15,	-- blacksmithing
			   ["DarkIronDwarf"] = 5,
			   ["KulTiran"] = 5},
	[45357] = {["Nightborne"] = 15,			-- inscription
			   ["KulTiran"] = 5},
	[2550]  = {["Pandaren"] = 15},			-- cooking
}

--
-- local function returns any racial bonus that may apply
--
local function getRacialBonus()
	local player = Skillet.currentPlayer
	local trade = Skillet.currentTrade
	local race = Skillet.db.realm.race[player]
	--DA.DEBUG(0,"getRacialBonus: trade = "..tostring(trade).." ("..tostring(GetSpellInfo(trade)).."), race= "..tostring(race))
	if racialList[trade] then
		for r, bonus in pairs(racialList[trade]) do
			--DA.DEBUG(1,"getRacialBonus: r = "..tostring(r)..", bonus= "..tostring(bonus))
			if r == race then
			DA.DEBUG(1,"getRacialBonus: bonus = "..tostring(bonus))
				return bonus
			end
		end
	end
	return 0
end

--
-- Get TradeSkill Difficulty Levels
--
-- Note: MainFrame.lua uses both inputs, other calls just use itemID.
--   Old MainFrame.lua call: GetTradeSkillLevels((recipe.itemID>0 and recipe.itemID) or -recipe.spellID)
--
function Skillet:GetTradeSkillLevels(itemID, spellID)
	--DA.DEBUG(0,"GetTradeSkillLevels("..tostring(itemID)..", "..tostring(spellID)..")")
	local a,b,c,d
	local rb = getRacialBonus()
	local skillLevels = Skillet.db.global.SkillLevels
	if itemID then 
		if tonumber(itemID) ~= nil and itemID ~= 0 then
			if self.isCraft or self.currentTrade == 7411 then  -- isCraft for Classic and BCC, Enchanting TradeID for retail
				itemID = -itemID
			end
			self.indexTradeSkillLevel = itemID
--
-- If there is an entry in our own table, use it
--
			if skillLevels and skillLevels[itemID] then
				local levels = skillLevels[itemID]
				if type(levels) == 'table' then
					if spellID then
						if isRetail then
							levels = skillLevels[itemID][spellID]
						else
							for spell, strng in pairs(levels) do
								name = GetSpellInfo(spell)
								--DA.DEBUG(1,"GetTradeSkillLevels: name= "..tostring(name))
								if name == spellID then
									levels = strng
									break
								end
							end
						end
					end
				end
				--DA.DEBUG(1,"GetTradeSkillLevels: levels= "..tostring(levels))
				if levels and type(levels) == 'string' then
					a,b,c,d = string.split("/", levels)
					a = (tonumber(a) or 0) + rb
					b = (tonumber(b) or 0) + rb
					c = (tonumber(c) or 0) + rb
					d = (tonumber(d) or 0) + rb
					self.sourceTradeSkillLevel = 1
					return a, b, c, d
				end
			end

--
-- The TradeskillInfo addon seems to be more accurate than LibPeriodicTable-3.1
--
			if isRetail and TradeskillInfo then
				local recipeSource = Skillet.db.global.itemRecipeSource[itemID]
				if not recipeSource then
					--DA.DEBUG(1,"GetTradeSkillLevels: itemID= "..tostring(itemID)..", recipeSource= "..tostring(recipeSource))
					recipeSource = Skillet.db.global.itemRecipeSource[-itemID]
				end
				if type(recipeSource) == 'table' then
					--DA.DEBUG(1,"GetTradeSkillLevels: itemID= "..tostring(itemID)..", recipeSource= "..DA.DUMP1(recipeSource))
					for recipeID in pairs(recipeSource) do
						--DA.DEBUG(2,"GetTradeSkillLevels: recipeID= "..tostring(recipeID))
						local TSILevels = TradeskillInfo:GetCombineDifficulty(recipeID)
						if type(TSILevels) == 'table' then
							--DA.DEBUG(2,"GetTradeSkillLevels: TSILevels="..DA.DUMP1(TSILevels))
							a = (tonumber(TSILevels[1]) or 0) + rb
							b = (tonumber(TSILevels[2]) or 0) + rb
							c = (tonumber(TSILevels[3]) or 0) + rb
							d = (tonumber(TSILevels[4]) or 0) + rb
							self.sourceTradeSkillLevel = 2
							return a, b, c, d
						end
					end
				else
					--DA.DEBUG(1,"GetTradeSkillLevels: itemID= "..tostring(itemID)..", recipeSource= "..tostring(recipeSource))
				end
			end

--
-- Check LibPeriodicTable
-- Note: The itemID for Enchants is negative
--
			if PT then
				local levels = PT:ItemInSet(itemID,"TradeskillLevels")
				--DA.DEBUG(1,"GetTradeSkillLevels (PT): itemID= "..tostring(itemID)..", levels= "..tostring(levels))
				if not levels then
					itemID = -itemID
					levels = PT:ItemInSet(itemID,"TradeskillLevels")
					--DA.DEBUG(1,"GetTradeSkillLevels (PT): itemID= "..tostring(itemID)..", levels= "..tostring(levels))
				end
				if levels then
					a,b,c,d = string.split("/",levels)
					a = (tonumber(a) or 0) + rb
					b = (tonumber(b) or 0) + rb
					c = (tonumber(c) or 0) + rb
					d = (tonumber(d) or 0) + rb
					self.sourceTradeSkillLevel = 3
					return a, b, c, d
				end
			end
		else
			DA.DEBUG(0,"GetTradeSkillLevels: "..tostring(itemID).." is not a number")
			self.sourceTradeSkillLevel = 4
			self.indexTradeSkillLevel = nil
			return 0, 0, 0, 0 
		end
	else
		DA.DEBUG(0,"GetTradeSkillLevels: itemID is missing")
		self.sourceTradeSkillLevel = 4
		self.indexTradeSkillLevel = nil
		return 0, 0, 0, 0 
	end
	if not self.db.global.MissingSkillLevels then
		self.db.global.MissingSkillLevels = {}
	end
	self.db.global.MissingSkillLevels[itemID] = "0/0/0/0"
	self.sourceTradeSkillLevel = 5
	return 0, 0, 0, 0 
end

function Skillet:GetTradeSkillLevelColor(itemID, rank)
	--DA.DEBUG(0,"GetTradeSkillLevelColor("..tostring(itemID)..", "..tostring(rank)")")
	if itemID then
		local orange, yellow, green, gray = self:GetTradeSkillLevels(itemID)
		if rank >= gray then return skillColors["trivial"] end
		if rank >= green then return skillColors["easy"] end
		if rank >= yellow then return skillColors["moderate"] end
		if rank >= orange then return skillColors["optimal"] end
	end
	return skillColors["unknown"]
end

function Skillet:AddTradeSkillLevels(itemID, orange, yellow, green, gray, spellID)
	DA.DEBUG(0,"AddTradeSkillLevels("..tostring(itemID)..", "..tostring(orange)..", "..tostring(yellow)..", "..tostring(green)..", "..tostring(gray)..", "..tostring(spellID)..")")
	local skillLevels = Skillet.db.global.SkillLevels
--
-- We should add some sanity checking
--
	if itemID and spellID then
		if not skillLevels[itemID] then 
			skillLevels[itemID] = {}
		end
		skillLevels[itemID][spellID] = tostring(orange).."/"..tostring(yellow).."/"..tostring(green).."/"..tostring(gray)
	elseif itemID then
		skillLevels[itemID] = tostring(orange).."/"..tostring(yellow).."/"..tostring(green).."/"..tostring(gray)
	end
end

function Skillet:DelTradeSkillLevels(itemID)
	DA.DEBUG(0,"DelTradeSkillLevels("..tostring(itemID)..")")
	local skillLevels = Skillet.db.global.SkillLevels
	if itemID then
--
-- We could add some additional checking
--
		skillLevels[itemID] = nil
	end
end

--
-- Print the TradeSkillLevels(itemID) result including the actual index and the source.
--
-- index will be itemID or if the current profession is Enchanting, -itemID
--
-- source will be:
--    1 if from Skillet.db.global.SkillLevels
--    2 if from TradeskillInfo
--    3 if from LibPeriodicTable
--    4 if itemID was missing or not a number
--    5 if it wasn't found (and was added to Skillet.db.global.MissingSkillLevels)
--
function Skillet:PrintTradeSkillLevels(itemID, spellID)
	--DA.DEBUG(0,"PrintTradeSkillLevels("..tostring(itemID)..", "..tostring(spellID)..")")
	if itemID then
		local orange, yellow, green, gray = self:GetTradeSkillLevels(itemID, spellID)
		DA.CHAT("PrintTradeSkillLevels: itemID= "..tostring(itemID)..", spellID= "..tostring(spellID))
		DA.CHAT("PrintTradeSkillLevels: index= "..tostring(self.indexTradeSkillLevel)..", source= "..tostring(self.sourceTradeSkillLevel))
		DA.CHAT("PrintTradeSkillLevels: levels= "..tostring(orange).."/"..tostring(yellow).."/"..tostring(green).."/"..tostring(gray))
	end
end

--
-- Our own table of skill levels (Skillet.db.global.SkillLevels)
--   1) can be maintained manually
--      To scrape from Wowhead: Follow instructions in ScrapeWowhead.js
--
--   2) can be maintained with an external addon (using AddTradeSkillLevels, DelTradeSkillLevels)
--
-- Each entry is either [itemID] = "orange/yellow/green/gray" or 
--   a table of spellIDs with the same format
--   [itemID] = {[spellID1] = "orange/yellow/green/gray", [spellID2] = "orange/yellow/green/gray"},
--
--   itemID is the item made by the recipe or the recipeID of the Enchant
--   spellID(s) are the numeric recipeID(s) of each recipe that produces the itemID
--   orange is the (numeric) skill level below which recipe is orange
--   yellow is the (numeric) skill level below which the recipe is yellow
--   green  is the (numeric) skill level below which the recipe is green
--   gray   is the (numeric) skill level above which the recipe is gray
--
-- A global table, Skillet.db.global.MissingSkillLevels, is added to when
-- no other entry is found. The format of this table is the same as
-- the table Skillet.db.global.SkillLevels to facilitate adding to this table.
--
-- InitializeSkillLevels is only called if the table does not exist.
--
-- Force refresh with '/skillet initskilllevels'
-- or '/run Skillet:InitializeSkillLevels()'
-- or '/run Skillet.db.global.SkillLevels = nil'
--
function Skillet:InitializeSkillLevels()
	self.db.global.SkillLevels = {
		[0] = "orange/yellow/green/gray",
		-- alchemy
		[118] = "1/55/75/95",
		[858] = "55/85/105/125",
		[929] = "110/135/155/175",
		[1710] = "155/175/195/215",
		[2454] = "1/55/75/95",
		[2455] = "25/65/85/105",
		[2456] = "40/70/90/110",
		[2457] = "50/80/100/120",
		[2458] = "50/80/100/120",
		[2459] = "60/90/110/130",
		[3382] = "15/60/80/100",
		[3383] = "90/120/140/160",
		[3384] = "110/135/155/175",
		[3385] = "120/145/165/185",
		[3386] = "120/145/165/185",
		[3387] = "250/275/295/315",
		[3388] = "125/150/170/190",
		[3389] = "130/155/175/195",
		[3390] = "140/165/185/205",
		[3391] = "150/175/195/215",
		[3577] = "225/240/260/280",
		[3823] = "165/185/205/225",
		[3824] = "165/190/210/230",
		[3825] = "175/195/215/235",
		[3826] = "180/200/220/240",
		[3827] = "160/180/200/220",
		[3828] = "195/215/235/255",
		[3829] = "200/220/240/260",
		[3928] = "215/230/250/270",
		[4596] = "50/80/100/120",
		[4623] = "215/230/250/270",
		[5631] = "60/90/110/130",
		[5633] = "175/195/215/235",
		[5634] = "150/175/195/215",
		[5996] = "90/120/140/160",
		[5997] = "1/55/75/95",
		[6037] = "225/240/260/280",
		[6048] = "135/160/180/200",
		[6049] = "165/210/230/250",
		[6050] = "190/205/225/245",
		[6051] = "100/130/150/170",
		[6052] = "190/210/230/250",
		[6149] = "205/220/240/260",
		[6370] = "80/80/90/100",
		[6371] = "130/150/160/170",
		[6372] = "100/130/150/170",
		[6373] = "140/165/185/205",
		[6662] = "90/120/140/160",
		[7068] = "300/301/305/310",
		[7076] = {[17560] = "275/275/282/290", [17565] = "275/275/282/290"},
		[7078] = "275/275/282/290",
		[7080] = {[17561] = "275/275/282/290", [17563] = "275/275/282/290"},
		[7082] = "275/275/282/290",
		[8949] = "185/205/225/245",
		[8951] = "195/215/235/255",
		[8956] = "205/220/240/260",
		[9030] = "215/225/245/265",
		[9036] = "210/225/245/265",
		[9061] = "210/225/245/265",
		[9088] = "240/255/275/295",
		[9144] = "225/240/260/280",
		[9149] = "225/240/260/280",
		[9154] = "230/245/265/285",
		[9155] = "235/250/270/290",
		[9172] = "235/250/270/290",
		[9179] = "235/250/270/290",
		[9187] = "240/255/275/295",
		[9197] = "240/255/275/295",
		[9206] = "245/260/280/300",
		[9210] = "245/260/280/300",
		[9224] = "250/265/285/305",
		[9233] = "250/265/285/305",
		[9264] = "250/265/285/305",
		[10592] = "200/220/240/260",
		[12190] = "230/245/265/285",
		[12360] = "275/275/282/290",
		[12803] = "275/275/282/290",
		[12808] = "275/275/282/290",
		[13423] = "250/250/255/260",
		[13442] = "255/270/290/310",
		[13443] = "260/275/295/315",
		[13444] = "295/310/330/350",
		[13445] = "265/280/300/320",
		[13446] = "275/290/310/330",
		[13447] = "270/285/305/325",
		[13452] = "280/295/315/335",
		[13453] = "275/290/310/330",
		[13454] = "285/300/320/340",
		[13455] = "280/295/315/335",
		[13456] = "290/305/325/345",
		[13457] = "290/305/325/345",
		[13458] = "290/305/325/345",
		[13459] = "290/305/325/345",
		[13461] = "290/305/325/345",
		[13462] = "285/300/320/340",
		[13503] = "300/365/372/380",
		[13506] = "300/315/322/330",
		[13510] = "300/315/322/330",
		[13511] = "300/315/322/330",
		[13512] = "300/315/322/330",
		[13513] = "300/315/322/330",
		[17708] = "190/210/230/250",
		[18253] = "300/310/320/330",
		[18294] = "215/230/250/270",
		[19931] = "300/315/322/330",
		[20002] = "275/290/310/330",
		[20004] = "290/305/325/345",
		[20007] = "275/290/310/330",
		[20008] = "285/300/320/340",
		[21546] = "250/265/285/305",
		[21884] = {[28566] = "350/365/372/380", [28582] = "300/385/392/400"},
		[21885] = {[28567] = "350/365/372/380", [28580] = "300/385/392/400"},
		[21886] = "300/385/392/400",
		[22451] = "350/365/372/380",
		[22452] = {[28568] = "350/365/372/380", [28584] = "300/385/392/400"},
		[22456] = "300/385/392/400",
		[22457] = "300/385/392/400",
		[22823] = "305/320/327/335",
		[22824] = "305/320/327/335",
		[22825] = "310/325/332/340",
		[22826] = "315/330/337/345",
		[22827] = "320/335/342/350",
		[22828] = "320/335/342/350",
		[22829] = "325/340/347/355",
		[22830] = "325/340/347/355",
		[22831] = "330/345/352/360",
		[22832] = "340/355/362/370",
		[22833] = "345/360/367/375",
		[22834] = "345/360/367/375",
		[22835] = "350/365/372/380",
		[22836] = "350/365/372/380",
		[22837] = "350/365/372/380",
		[22838] = "350/365/372/380",
		[22839] = "350/365/372/380",
		[22840] = "355/370/377/385",
		[22841] = "360/375/382/390",
		[22842] = "360/375/382/390",
		[22844] = "360/375/382/390",
		[22845] = "360/375/382/390",
		[22846] = "360/375/382/390",
		[22847] = "360/375/382/390",
		[22848] = "365/380/387/395",
		[22849] = "365/380/387/395",
		[22850] = "300/390/397/405",
		[22851] = "300/390/397/405",
		[22853] = "300/390/397/405",
		[22854] = "300/390/397/405",
		[22861] = "300/390/397/405",
		[22866] = "300/390/397/405",
		[22871] = "335/350/357/365",
		[23571] = "350/365/372/380",
		[25867] = "350/365/372/380",
		[25868] = "350/365/372/380",
		[28100] = "300/315/322/330",
		[28101] = "310/325/332/340",
		[28102] = "300/315/322/330",
		[28103] = "300/315/322/330",
		[28104] = "315/330/337/345",
		[31080] = "325/340/347/355",
		[31676] = "345/360/367/375",
		[31677] = "360/375/382/390",
		[31679] = "335/350/357/365",
		[32062] = "310/325/332/340",
		[32063] = "320/335/342/350",
		[32067] = "320/335/342/350",
		[32068] = "330/345/352/360",
		[32839] = "360/360/370/380",
		[32849] = "360/360/370/380",
		[32850] = "360/360/370/380",
		[32851] = "360/360/370/380",
		[32852] = "360/360/370/380",
		[33208] = "375/390/397/405",
		[34440] = "325/335/342/350",
		[35748] = "375/390/397/405",
		[35749] = "375/390/397/405",
		[35750] = "375/390/397/405",
		[35751] = "375/390/397/405",
		
		-- blacksmithing
		[2844] = "15/55/75/95",
		[2845] = "20/60/80/100",
		[2847] = "25/65/85/105",
		[2848] = "110/140/155/170",
		[2849] = "115/145/160/175",
		[2850] = "120/150/165/180",
		[2851] = "35/75/95/115",
		[2852] = "1/50/70/90",
		[2853] = "1/20/40/60",
		[2854] = "90/115/127/140",
		[2857] = "70/110/130/150",
		[2862] = "1/15/35/55",
		[2863] = "65/65/72/80",
		[2864] = "80/120/140/160",
		[2865] = "105/145/160/175",
		[2866] = "105/145/160/175",
		[2868] = "120/150/165/180",
		[2869] = "130/160/175/190",
		[2870] = "145/175/190/205",
		[2871] = "125/125/132/140",
		[3239] = "1/15/35/55",
		[3240] = "65/65/72/80",
		[3241] = "125/125/132/140",
		[3469] = "20/60/80/100",
		[3470] = "25/45/65/85",
		[3471] = "35/75/95/115",
		[3472] = "40/80/100/120",
		[3473] = "45/85/105/125",
		[3474] = "60/100/120/140",
		[3478] = "75/75/87/100",
		[3480] = "110/140/155/170",
		[3481] = "125/155/170/185",
		[3482] = "130/160/175/190",
		[3483] = "135/165/180/195",
		[3484] = "145/175/190/205",
		[3485] = "150/180/195/210",
		[3486] = "125/125/137/150",
		[3487] = "95/135/155/175",
		[3488] = "35/75/95/115",
		[3489] = "70/110/130/150",
		[3490] = "125/155/170/185",
		[3491] = "130/160/175/190",
		[3492] = "145/175/190/205",
		[3835] = "165/190/202/215",
		[3836] = "170/195/207/220",
		[3837] = "190/215/227/240",
		[3840] = "160/185/197/210",
		[3841] = "175/200/212/225",
		[3842] = "155/180/192/205",
		[3843] = "170/195/207/220",
		[3844] = "180/205/217/230",
		[3845] = "195/220/232/245",
		[3846] = "185/210/222/235",
		[3847] = "200/225/237/250",
		[3848] = "105/135/150/165",
		[3849] = "160/185/197/210",
		[3850] = "175/200/212/225",
		[3851] = "155/180/192/205",
		[3852] = "170/195/207/220",
		[3853] = "180/205/217/230",
		[3854] = "200/225/237/250",
		[3855] = "185/210/222/235",
		[3856] = "200/225/237/250",
		[5540] = "110/140/155/170",
		[5541] = "140/170/185/200",
		[6040] = "185/210/222/235",
		[6041] = "190/215/227/240",
		[6042] = "150/180/195/210",
		[6043] = "165/190/202/215",
		[6214] = "65/105/125/145",
		[6338] = "100/105/107/110",
		[6350] = "95/125/140/155",
		[6731] = "100/140/160/180",
		[7071] = "150/150/152/155",
		[7166] = "30/70/90/110",
		[7913] = "160/185/197/210",
		[7914] = "160/185/197/210",
		[7915] = "175/200/212/225",
		[7916] = "180/205/217/230",
		[7917] = "185/210/222/235",
		[7918] = "205/225/235/245",
		[7919] = "205/225/235/245",
		[7920] = "210/230/240/250",
		[7921] = "210/230/240/250",
		[7922] = "215/235/245/255",
		[7924] = "215/235/245/255",
		[7926] = "220/240/250/260",
		[7927] = "220/240/250/260",
		[7928] = "225/245/255/265",
		[7929] = "230/250/260/270",
		[7930] = "230/250/260/270",
		[7931] = "230/250/260/270",
		[7932] = "235/255/265/275",
		[7933] = "235/255/265/275",
		[7934] = "245/255/265/275",
		[7935] = "240/260/270/280",
		[7936] = "245/265/275/285",
		[7937] = "245/265/275/285",
		[7938] = "225/245/255/265",
		[7939] = "245/265/275/285",
		[7941] = "210/235/247/260",
		[7942] = "220/245/257/270",
		[7943] = "225/250/262/275",
		[7944] = "240/265/277/290",
		[7945] = "230/255/267/280",
		[7946] = "245/270/282/295",
		[7947] = "255/280/292/305",
		[7954] = "235/260/272/285",
		[7955] = "30/70/90/110",
		[7956] = "125/155/170/185",
		[7957] = "130/160/175/190",
		[7958] = "135/165/180/195",
		[7959] = "250/275/287/300",
		[7960] = "260/285/297/310",
		[7961] = "245/270/282/295",
		[7963] = "200/225/237/250",
		[7964] = "200/200/205/210",
		[7965] = "200/200/205/210",
		[7966] = "200/200/205/210",
		[7967] = "215/235/245/255",
		[7969] = "235/255/265/275",
		[9060] = "200/225/237/250",
		[9366] = "205/225/235/245",
		[10421] = "1/15/35/55",
		[10423] = "155/180/192/205",
		[11128] = "150/155/157/160",
		[11144] = "200/205/207/210",
		[11604] = "285/305/315/325",
		[11605] = "280/300/310/320",
		[11606] = "270/290/300/310",
		[11607] = "275/295/305/315",
		[11608] = "265/285/295/305",
		[12259] = "180/205/217/230",
		[12260] = "190/215/227/240",
		[12404] = "250/255/257/260",
		[12405] = "250/270/280/290",
		[12406] = "250/270/280/290",
		[12408] = "255/275/285/295",
		[12409] = "280/300/310/320",
		[12410] = "280/300/310/320",
		[12414] = "300/320/330/340",
		[12415] = "270/290/300/310",
		[12416] = "260/280/290/300",
		[12417] = "295/315/325/335",
		[12418] = "285/305/315/325",
		[12419] = "290/310/320/330",
		[12420] = "300/320/330/340",
		[12422] = "300/320/330/340",
		[12424] = "265/285/295/305",
		[12425] = "270/290/300/310",
		[12426] = "295/315/325/335",
		[12427] = "295/315/325/335",
		[12428] = "265/285/295/305",
		[12429] = "300/320/330/340",
		[12610] = "300/320/330/340",
		[12611] = "300/320/330/340",
		[12612] = "300/320/330/340",
		[12613] = "300/320/330/340",
		[12614] = "300/320/330/340",
		[12618] = "300/320/330/340",
		[12619] = "300/320/330/340",
		[12620] = "300/320/330/340",
		[12624] = "270/290/300/310",
		[12625] = "290/310/320/330",
		[12628] = "285/305/315/325",
		[12631] = "290/310/320/330",
		[12632] = "295/315/325/335",
		[12633] = "300/320/330/340",
		[12636] = "300/320/330/340",
		[12639] = "300/320/330/340",
		[12640] = "300/320/330/340",
		[12641] = "300/320/330/340",
		[12643] = "250/255/257/260",
		[12644] = "250/255/257/260",
		[12645] = "275/295/305/315",
		[12773] = "275/300/312/325",
		[12774] = "275/300/312/325",
		[12775] = "280/305/317/330",
		[12776] = "280/305/317/330",
		[12777] = "280/305/317/330",
		[12781] = "285/310/322/335",
		[12782] = "290/315/327/340",
		[12783] = "300/320/330/340",
		[12784] = "300/320/330/340",
		[12790] = "300/320/330/340",
		[12792] = "290/315/327/340",
		[12794] = "300/320/330/340",
		[12796] = "300/320/330/340",
		[12797] = "300/320/330/340",
		[12798] = "300/320/330/340",
		[15869] = "100/100/110/120",
		[15870] = "150/150/160/170",
		[15871] = "200/200/210/220",
		[15872] = "275/275/280/285",
		[16206] = "275/275/280/285",
		[16988] = "300/320/330/340",
		[16989] = "295/315/325/335",
		[17013] = "300/320/330/340",
		[17014] = "295/315/325/335",
		[17015] = "300/320/330/340",
		[17016] = "300/320/330/340",
		[17193] = "300/325/337/350",
		[17704] = "190/215/227/240",
		[18262] = "300/300/310/320",
		[19043] = "290/310/320/330",
		[19048] = "300/320/330/340",
		[19051] = "290/310/320/330",
		[19057] = "300/320/330/340",
		[19148] = "300/320/330/340",
		[19164] = "300/320/330/340",
		[19166] = "300/320/330/340",
		[19167] = "300/320/330/340",
		[19168] = "300/320/330/340",
		[19169] = "300/320/330/340",
		[19170] = "300/320/330/340",
		[19690] = "300/320/330/340",
		[19691] = "300/320/330/340",
		[19692] = "300/320/330/340",
		[19693] = "300/320/330/340",
		[19694] = "300/320/330/340",
		[19695] = "300/320/330/340",
		[20039] = "300/320/330/340",
		[20549] = "300/320/330/340",
		[20550] = "300/320/330/340",
		[20551] = "300/320/330/340",
		[22191] = "300/320/330/340",
		[22194] = "300/320/330/340",
		[22195] = "300/320/330/340",
		[22196] = "300/320/330/340",
		[22197] = "300/320/330/340",
		[22198] = "300/320/330/340",
		[22383] = "300/320/330/340",
		[22384] = "300/320/330/340",
		[22385] = "300/320/330/340",
		[22669] = "300/320/330/340",
		[22670] = "300/320/330/340",
		[22671] = "300/320/330/340",
		[22762] = "300/320/330/340",
		[22763] = "300/320/330/340",
		[22764] = "300/320/330/340",
		[23482] = "300/310/320/330",
		[23484] = "305/315/325/335",
		[23487] = "315/325/335/345",
		[23488] = "315/325/335/345",
		[23489] = "325/335/345/355",
		[23490] = "320/330/340/350",
		[23491] = "310/320/330/340",
		[23493] = "300/310/320/330",
		[23494] = "315/325/335/345",
		[23497] = "310/320/330/340",
		[23498] = "315/325/335/345",
		[23499] = "320/330/340/350",
		[23502] = "325/335/345/355",
		[23503] = "330/340/350/360",
		[23504] = "330/340/350/360",
		[23505] = "335/345/355/365",
		[23506] = "335/345/355/365",
		[23507] = "340/350/360/370",
		[23508] = "335/345/355/365",
		[23509] = "360/370/380/390",
		[23510] = "355/365/375/385",
		[23511] = "355/365/375/385",
		[23512] = "365/375/385/395",
		[23513] = "365/375/385/395",
		[23514] = "360/370/380/390",
		[23515] = "350/360/370/380",
		[23516] = "355/365/375/385",
		[23517] = "360/370/380/390",
		[23518] = "360/370/380/390",
		[23519] = "365/375/385/395",
		[23520] = "365/375/385/395",
		[23521] = "365/375/385/395",
		[23522] = "370/380/390/400",
		[23523] = "360/370/380/390",
		[23524] = "360/370/380/390",
		[23525] = "365/375/385/395",
		[23526] = "370/380/390/400",
		[23527] = "370/380/390/400",
		[23528] = "300/300/305/310",
		[23529] = "350/350/355/360",
		[23530] = "360/370/380/390",
		[23531] = "365/375/385/395",
		[23532] = "365/375/385/395",
		[23533] = "365/375/385/395",
		[23534] = "365/375/385/395",
		[23535] = "365/375/385/395",
		[23536] = "365/375/385/395",
		[23537] = "365/375/385/395",
		[23538] = "365/375/385/395",
		[23539] = "365/375/385/395",
		[23540] = "365/375/385/395",
		[23541] = "365/375/385/395",
		[23542] = "365/375/385/395",
		[23543] = "365/375/385/395",
		[23544] = "365/375/385/395",
		[23546] = "365/375/385/395",
		[23554] = "365/375/385/395",
		[23555] = "365/375/385/395",
		[23556] = "365/375/385/395",
		[23559] = "325/325/330/335",
		[23563] = "350/360/370/380",
		[23564] = "375/385/395/405",
		[23565] = "375/385/395/405",
		[23575] = "340/340/345/350",
		[23576] = "375/375/375/375",
		[25521] = "350/350/355/360",
		[25843] = "300/300/305/310",
		[25844] = "350/350/355/360",
		[25845] = "375/375/380/385",
		[28420] = "300/300/305/310",
		[28421] = "350/350/355/360",
		[28425] = "350/360/370/380",
		[28426] = "375/385/395/405",
		[28427] = "375/385/395/405",
		[28428] = "350/360/370/380",
		[28429] = "375/385/395/405",
		[28430] = "375/385/395/405",
		[28431] = "350/360/370/380",
		[28432] = "375/385/395/405",
		[28433] = "375/385/395/405",
		[28434] = "350/360/370/380",
		[28435] = "375/385/395/405",
		[28436] = "375/385/395/405",
		[28437] = "350/360/370/380",
		[28438] = "375/385/395/405",
		[28439] = "375/385/395/405",
		[28440] = "350/360/370/380",
		[28441] = "375/385/395/405",
		[28442] = "375/385/395/405",
		[28483] = "350/360/370/380",
		[28484] = "375/385/395/405",
		[28485] = "375/385/395/405",
		[29201] = "100/130/145/160",
		[29202] = "200/220/230/240",
		[29203] = "300/320/330/340",
		[29204] = "350/360/370/380",
		[30031] = "375/385/395/405",
		[30032] = "375/385/395/405",
		[30033] = "375/385/395/405",
		[30034] = "375/385/395/405",
		[30069] = "260/280/290/300",
		[30070] = "260/280/290/300",
		[30071] = "260/280/290/300",
		[30072] = "260/280/290/300",
		[30073] = "260/280/290/300",
		[30074] = "330/340/350/360",
		[30076] = "330/340/350/360",
		[30077] = "330/340/350/360",
		[30086] = "330/340/350/360",
		[30087] = "330/340/350/360",
		[30088] = "330/340/350/360",
		[30089] = "330/340/350/360",
		[30093] = "330/340/350/360",
		[31364] = "375/385/395/405",
		[31367] = "375/385/395/405",
		[31368] = "375/385/395/405",
		[31369] = "375/385/395/405",
		[31370] = "375/385/395/405",
		[31371] = "375/385/395/405",
		[32401] = "375/385/395/405",
		[32402] = "375/385/395/405",
		[32403] = "375/385/395/405",
		[32404] = "375/385/395/405",
		[32568] = "375/385/395/405",
		[32570] = "375/385/395/405",
		[32571] = "375/385/395/405",
		[32573] = "375/385/395/405",
		[32854] = "365/375/385/395",
		[33173] = "365/375/385/395",
		[33185] = "335/345/350/355",
		[33791] = "35/75/95/115",
		[34377] = "365/375/392/410",
		[34378] = "365/375/392/410",
		[34379] = "365/375/392/410",
		[34380] = "365/375/392/410",
		
		-- enchanting
		[-7418] = "1/70/90/110",
		[-7420] = "15/70/90/110",
		[-7421] = "1/5/7/10",
		[-7426] = "40/90/110/130",
		[-7428] = "1/80/100/120",
		[-7443] = "20/80/100/120",
		[-7454] = "45/95/115/135",
		[-7457] = "50/100/120/140",
		[-7745] = "100/130/150/170",
		[-7748] = "60/105/125/145",
		[-7766] = "60/105/125/145",
		[-7771] = "70/110/130/150",
		[-7776] = "80/115/135/155",
		[-7779] = "80/115/135/155",
		[-7782] = "80/115/135/155",
		[-7786] = "90/120/140/160",
		[-7788] = "90/120/140/160",
		[-7793] = "100/130/150/170",
		[-7795] = "100/130/150/170",
		[-7857] = "120/145/165/185",
		[-7859] = "120/145/165/185",
		[-7861] = "125/150/170/190",
		[-7863] = "125/150/170/190",
		[-7867] = "125/150/170/190",
		[-13378] = "105/130/150/170",
		[-13380] = "110/135/155/175",
		[-13419] = "110/135/155/175",
		[-13421] = "115/140/160/180",
		[-13464] = "115/140/160/180",
		[-13485] = "130/155/175/195",
		[-13501] = "130/155/175/195",
		[-13503] = "140/165/185/205",
		[-13522] = "135/160/180/200",
		[-13529] = "145/170/190/210",
		[-13536] = "140/165/185/205",
		[-13538] = "140/165/185/205",
		[-13607] = "145/170/190/210",
		[-13612] = "145/170/190/210",
		[-13617] = "145/170/190/210",
		[-13620] = "145/170/190/210",
		[-13622] = "150/175/195/215",
		[-13626] = "150/175/195/215",
		[-13628] = "150/175/195/215",
		[-13631] = "155/175/195/215",
		[-13635] = "155/175/195/215",
		[-13637] = "160/180/200/220",
		[-13640] = "160/180/200/220",
		[-13642] = "165/185/205/225",
		[-13644] = "170/190/210/230",
		[-13646] = "170/190/210/230",
		[-13648] = "170/190/210/230",
		[-13653] = "175/195/215/235",
		[-13655] = "175/195/215/235",
		[-13657] = "175/195/215/235",
		[-13659] = "180/200/220/240",
		[-13661] = "180/200/220/240",
		[-13663] = "185/205/225/245",
		[-13687] = "190/210/230/250",
		[-13689] = "195/215/235/255",
		[-13693] = "195/215/235/255",
		[-13695] = "200/220/240/260",
		[-13698] = "200/220/240/260",
		[-13700] = "200/220/240/260",
		[-13702] = "200/220/240/260",
		[-13746] = "205/225/245/265",
		[-13794] = "205/225/245/265",
		[-13815] = "210/230/250/270",
		[-13817] = "210/230/250/270",
		[-13822] = "210/230/250/270",
		[-13836] = "215/235/255/275",
		[-13841] = "215/235/255/275",
		[-13846] = "220/240/260/280",
		[-13858] = "220/240/260/280",
		[-13868] = "225/245/265/285",
		[-13882] = "225/245/265/285",
		[-13887] = "225/245/265/285",
		[-13890] = "225/245/265/285",
		[-13898] = "265/285/305/325",
		[-13905] = "230/250/270/290",
		[-13915] = "230/250/270/290",
		[-13917] = "230/250/270/290",
		[-13931] = "235/255/275/295",
		[-13933] = "235/255/275/295",
		[-13935] = "235/255/275/295",
		[-13937] = "240/260/280/300",
		[-13939] = "240/260/280/300",
		[-13941] = "245/265/285/305",
		[-13943] = "245/265/285/305",
		[-13945] = "245/265/285/305",
		[-13947] = "250/270/290/310",
		[-13948] = "250/270/290/310",
		[-14293] = "10/75/95/115",
		[-14807] = "70/110/130/150",
		[-14809] = "155/175/195/215",
		[-14810] = "175/195/215/235",
		[-15596] = "265/285/305/325",
		[-17180] = "250/250/255/260",
		[-17181] = "250/250/255/260",
		[-20008] = "255/275/295/315",
		[-20009] = "270/290/310/330",
		[-20010] = "295/310/325/340",
		[-20011] = "300/310/325/340",
		[-20012] = "270/290/310/330",
		[-20013] = "295/310/325/340",
		[-20014] = "265/285/305/325",
		[-20015] = "285/300/317/335",
		[-20016] = "280/300/320/340",
		[-20017] = "265/285/305/325",
		[-20020] = "260/280/300/320",
		[-20023] = "295/310/325/340",
		[-20024] = "275/295/315/335",
		[-20025] = "300/310/325/340",
		[-20026] = "275/295/315/335",
		[-20028] = "290/305/322/340",
		[-20029] = "285/300/317/335",
		[-20030] = "295/310/325/340",
		[-20031] = "300/310/325/340",
		[-20032] = "300/310/325/340",
		[-20033] = "295/310/325/340",
		[-20034] = "300/310/325/340",
		[-20035] = "300/310/325/340",
		[-20036] = "300/310/325/340",
		[-20051] = "290/305/322/340",
		[-21931] = "190/210/230/250",
		[-22749] = "300/310/325/340",
		[-22750] = "300/310/325/340",
		[-23799] = "290/305/322/340",
		[-23800] = "290/305/322/340",
		[-23801] = "290/305/322/340",
		[-23802] = "300/310/325/340",
		[-23803] = "300/310/325/340",
		[-23804] = "300/310/325/340",
		[-25072] = "300/310/325/340",
		[-25073] = "300/310/325/340",
		[-25074] = "300/310/325/340",
		[-25078] = "300/310/325/340",
		[-25079] = "300/310/325/340",
		[-25080] = "300/310/325/340",
		[-25081] = "300/310/325/340",
		[-25082] = "300/310/325/340",
		[-25083] = "300/310/325/340",
		[-25084] = "300/310/325/340",
		[-25086] = "300/310/325/340",
		[-25124] = "45/55/65/75",
		[-25125] = "150/160/170/180",
		[-25126] = "200/210/220/230",
		[-25127] = "250/260/270/280",
		[-25128] = "275/285/295/305",
		[-25129] = "300/310/320/330",
		[-25130] = "300/310/320/330",
		[-27837] = "290/305/322/340",
		[-27899] = "305/315/330/345",
		[-27905] = "315/325/340/355",
		[-27906] = "320/330/345/360",
		[-27911] = "325/335/350/365",
		[-27913] = "335/345/360/375",
		[-27914] = "350/360/375/390",
		[-27917] = "360/370/385/400",
		[-27920] = "360/370/385/400",
		[-27924] = "360/370/385/400",
		[-27926] = "370/380/395/410",
		[-27927] = "375/385/400/415",
		[-27944] = "310/320/335/350",
		[-27945] = "325/335/350/365",
		[-27946] = "340/350/365/380",
		[-27947] = "360/370/385/400",
		[-27948] = "305/315/330/345",
		[-27950] = "320/330/345/360",
		[-27951] = "340/350/365/380",
		[-27954] = "370/380/395/410",
		[-27957] = "315/325/340/355",
		[-27958] = "325/335/350/365",
		[-27960] = "345/355/370/385",
		[-27961] = "310/320/335/350",
		[-27962] = "330/340/355/370",
		[-27967] = "340/350/365/380",
		[-27968] = "340/350/365/380",
		[-27971] = "350/360/375/390",
		[-27972] = "350/360/375/390",
		[-27975] = "350/360/375/390",
		[-27977] = "360/370/385/400",
		[-27981] = "375/385/400/415",
		[-27982] = "375/385/400/415",
		[-27984] = "375/385/400/415",
		[-28003] = "360/370/385/400",
		[-28004] = "360/370/385/400",
		[-28016] = "310/310/320/330",
		[-28019] = "340/340/350/360",
		[-28022] = "335/335/335/335",
		[-28027] = "325/325/330/335",
		[-28028] = "350/360/375/390",
		[-32664] = "300/310/325/340",
		[-32665] = "350/360/375/390",
		[-32667] = "375/375/385/400",
		[-33990] = "320/330/345/360",
		[-33991] = "300/310/325/340",
		[-33992] = "345/355/370/385",
		[-33993] = "305/315/330/345",
		[-33994] = "360/370/385/400",
		[-33995] = "340/350/365/380",
		[-33996] = "310/320/335/350",
		[-33997] = "360/370/385/400",
		[-33999] = "350/360/375/390",
		[-34001] = "305/315/330/345",
		[-34002] = "300/310/325/340",
		[-34003] = "325/335/350/365",
		[-34004] = "310/320/335/350",
		[-34005] = "350/360/375/390",
		[-34006] = "350/360/375/390",
		[-34007] = "360/370/385/400",
		[-34008] = "360/370/385/400",
		[-34009] = "325/335/350/365",
		[-34010] = "350/360/375/390",
		[-42613] = "300/300/300/305",
		[-42615] = "335/335/335/335",
		[-42620] = "350/360/367/375",
		[-42974] = "375/385/400/415",
		[-44383] = "330/340/355/370",
		[-45765] = "375/375/375/375",
		[-46578] = "350/350/357/365",
		[-46594] = "360/370/385/400",
		[-47051] = "375/380/395/410",
		
		-- engineering
		[4357] = "1/20/30/40",
		[4358] = "1/30/45/60",
		[4359] = "30/45/52/60",
		[4360] = "30/60/75/90",
		[4361] = "50/80/95/110",
		[4362] = "50/80/95/110",
		[4363] = "65/95/110/125",
		[4364] = "75/85/90/95",
		[4365] = "75/90/97/105",
		[4366] = "85/115/130/145",
		[4367] = "100/130/145/160",
		[4368] = "100/130/145/160",
		[4369] = "105/130/142/155",
		[4370] = "105/105/130/155",
		[4371] = "105/105/130/155",
		[4372] = "120/145/157/170",
		[4373] = "120/145/157/170",
		[4374] = "120/120/145/170",
		[4375] = "125/125/150/175",
		[4376] = "125/125/150/175",
		[4377] = "125/125/135/145",
		[4378] = "125/125/135/145",
		[4379] = "130/155/167/180",
		[4380] = "140/140/165/190",
		[4381] = "140/165/177/190",
		[4382] = "145/145/170/195",
		[4383] = "145/170/182/195",
		[4384] = "150/175/187/200",
		[4385] = "150/175/187/200",
		[4386] = "155/175/185/195",
		[4387] = "160/160/170/180",
		[4388] = "160/180/190/200",
		[4389] = "170/170/190/210",
		[4390] = "175/175/195/215",
		[4391] = "175/175/195/215",
		[4392] = "185/185/205/225",
		[4393] = "185/205/215/225",
		[4394] = "190/190/210/230",
		[4395] = "195/215/225/235",
		[4396] = "200/220/230/240",
		[4397] = "200/220/230/240",
		[4398] = "200/200/220/240",
		[4401] = "75/105/120/135",
		[4403] = "165/185/195/205",
		[4404] = "90/110/125/140",
		[4405] = "60/90/105/120",
		[4406] = "110/135/147/160",
		[4407] = "180/200/210/220",
		[4852] = "185/185/205/225",
		[5507] = "135/160/172/185",
		[6219] = "50/70/80/90",
		[6533] = "150/150/160/170",
		[6712] = "100/115/122/130",
		[6714] = "100/115/122/130",
		[7148] = "165/165/180/200",
		[7189] = "225/245/255/265",
		[7191] = "275/275/280/285",
		[7506] = "125/150/162/175",
		[8067] = "1/30/45/60",
		[8068] = "75/85/90/95",
		[8069] = "125/125/135/145",
		[9312] = "150/150/162/175",
		[9313] = "150/150/162/175",
		[9318] = "150/150/162/175",
		[10498] = "175/175/195/215",
		[10499] = "175/195/205/215",
		[10500] = "205/225/235/245",
		[10501] = "220/240/250/260",
		[10502] = "225/245/255/265",
		[10503] = "230/250/260/270",
		[10504] = "245/265/275/285",
		[10505] = "175/175/185/195",
		[10506] = "230/250/260/270",
		[10507] = "175/175/185/195",
		[10508] = "205/225/235/245",
		[10510] = "220/240/250/260",
		[10512] = "210/210/230/250",
		[10513] = "245/245/265/285",
		[10514] = "215/215/235/255",
		[10518] = "225/245/255/265",
		[10542] = "205/225/235/245",
		[10543] = "205/225/235/245",
		[10545] = "210/230/240/250",
		[10546] = "210/230/240/250",
		[10548] = "240/260/270/280",
		[10558] = "150/150/170/190",
		[10559] = "195/195/215/235",
		[10560] = "200/200/220/240",
		[10561] = "215/215/235/255",
		[10562] = "235/235/255/275",
		[10576] = "250/270/280/290",
		[10577] = {[12716] = "225/225/235/245", [13240] = "205/205/205/205"},
		[10586] = "235/235/255/275",
		[10587] = "230/230/250/270",
		[10588] = "245/265/275/285",
		[10644] = "205/205/205/205",
		[10645] = "240/260/270/280",
		[10646] = "205/205/225/245",
		[10713] = "205/205/205/205",
		[10716] = "205/225/235/245",
		[10720] = "210/230/240/250",
		[10721] = "215/235/245/255",
		[10724] = "225/245/255/265",
		[10725] = "230/250/260/270",
		[10726] = "235/255/265/275",
		[10727] = "240/260/270/280",
		[11590] = "200/200/220/240",
		[11825] = "205/205/205/205",
		[11826] = "205/205/205/205",
		[15846] = "250/270/280/290",
		[15992] = "250/250/255/260",
		[15993] = "260/280/290/300",
		[15994] = "260/280/290/300",
		[15995] = "260/280/290/300",
		[15996] = "265/285/295/305",
		[15997] = "285/295/300/305",
		[15999] = "270/290/300/310",
		[16000] = "275/295/305/315",
		[16004] = "275/295/305/315",
		[16005] = "285/305/315/325",
		[16006] = "285/305/315/325",
		[16007] = "300/320/330/340",
		[16008] = "290/310/320/330",
		[16009] = "290/310/320/330",
		[16022] = "300/320/330/340",
		[16023] = "275/295/305/315",
		[16040] = "300/320/330/340",
		[17716] = "190/190/210/230",
		[18168] = "300/320/330/340",
		[18232] = "300/320/330/340",
		[18282] = "300/320/330/340",
		[18283] = "300/320/330/340",
		[18587] = "265/285/295/305",
		[18588] = "200/200/210/220",
		[18594] = "275/275/285/295",
		[18631] = "260/270/275/280",
		[18634] = "260/280/290/300",
		[18637] = "275/285/290/295",
		[18638] = "290/310/320/330",
		[18639] = "300/320/330/340",
		[18641] = "250/250/260/270",
		[18645] = "265/275/280/285",
		[18660] = "260/260/265/270",
		[18984] = "285/285/295/305",
		[18986] = "285/285/295/305",
		[19026] = "250/250/260/270",
		[19998] = "300/320/330/340",
		[19999] = "300/320/330/340",
		[20475] = "335/335/345/355",
		[21277] = "250/320/330/340",
		[21557] = "125/125/137/150",
		[21558] = "125/125/137/150",
		[21559] = "125/125/137/150",
		[21569] = "225/245/255/265",
		[21570] = "275/295/305/315",
		[21571] = "225/225/237/250",
		[21574] = "225/225/237/250",
		[21576] = "225/225/237/250",
		[21589] = "175/175/187/200",
		[21590] = "175/175/187/200",
		[21592] = "175/175/187/200",
		[21714] = "275/275/280/285",
		[21716] = "275/275/280/285",
		[21718] = "275/275/280/285",
		[22728] = "275/275/280/285",
		[23736] = "300/320/330/340",
		[23737] = "325/335/345/355",
		[23742] = "320/330/340/350",
		[23746] = "350/360/370/380",
		[23747] = "360/370/380/390",
		[23748] = "375/385/395/405",
		[23758] = "340/350/360/370",
		[23761] = "340/350/360/370",
		[23762] = "350/360/370/380",
		[23763] = "360/370/380/390",
		[23764] = "335/345/355/365",
		[23765] = "360/370/380/390",
		[23766] = "375/385/395/405",
		[23767] = "325/335/345/355",
		[23768] = "335/335/345/355",
		[23771] = "335/335/345/355",
		[23772] = "310/310/320/330",
		[23774] = "325/325/335/345",
		[23775] = "350/360/370/380",
		[23781] = "300/300/310/320",
		[23782] = "300/300/310/320",
		[23783] = "300/300/305/310",
		[23784] = "325/325/330/335",
		[23785] = "340/350/360/370",
		[23786] = "340/350/360/370",
		[23787] = "340/350/360/370",
		[23819] = "350/350/355/360",
		[23821] = "305/305/315/325",
		[23824] = "355/365/375/385",
		[23825] = "350/360/370/380",
		[23826] = "325/325/335/345",
		[23827] = "340/340/350/360",
		[23828] = "375/375/385/395",
		[23829] = "375/375/385/395",
		[23835] = "340/360/370/380",
		[23836] = "350/360/370/380",
		[23838] = "375/375/385/395",
		[23839] = "375/375/385/395",
		[23841] = "325/335/345/355",
		[25886] = "335/335/345/355",
		[30542] = "350/350/360/370",
		[30544] = "350/350/360/370",
		[32413] = "335/345/355/365",
		[32423] = "335/335/340/345",
		[32461] = "370/370/380/390",
		[32472] = "370/370/380/390",
		[32473] = "370/370/380/390",
		[32474] = "370/370/380/390",
		[32475] = "370/370/380/390",
		[32476] = "370/370/380/390",
		[32478] = "370/370/380/390",
		[32479] = "370/370/380/390",
		[32480] = "370/370/380/390",
		[32494] = "370/370/380/390",
		[32495] = "370/370/380/390",
		[32756] = "375/375/392/410",
		[33092] = "330/330/340/350",
		[33093] = "345/345/355/365",
		[34060] = "350/375/380/385",
		[34061] = "375/385/390/395",
		[34113] = "360/370/380/390",
		[34353] = "375/390/410/430",
		[34354] = "375/390/410/430",
		[34355] = "375/390/410/430",
		[34356] = "375/390/410/430",
		[34357] = "375/390/410/430",
		[34504] = "335/335/345/355",
		[34847] = "375/390/410/430",
		[35181] = "375/390/410/430",
		[35182] = "375/390/410/430",
		[35183] = "375/390/410/430",
		[35184] = "375/390/410/430",
		[35185] = "375/390/410/430",
		[35581] = "355/365/375/385",
		
		-- jewelcrafting
		[20816] = "1/20/35/50",
		[20817] = "50/70/80/90",
		[20818] = "50/80/95/110",
		[20820] = "60/90/105/120",
		[20821] = "30/60/75/90",
		[20823] = "70/100/115/130",
		[20826] = "90/120/135/150",
		[20827] = "80/110/125/140",
		[20828] = "100/130/145/160",
		[20830] = "110/140/155/170",
		[20831] = "150/180/195/210",
		[20832] = "120/150/165/180",
		[20833] = "125/155/170/185",
		[20906] = "1/30/45/60",
		[20907] = "50/80/95/110",
		[20909] = "110/140/155/170",
		[20950] = "120/150/165/180",
		[20954] = "125/155/170/185",
		[20955] = "135/165/180/195",
		[20958] = "150/180/195/210",
		[20959] = "170/200/215/230",
		[20960] = "170/200/215/230",
		[20961] = "180/210/225/240",
		[20963] = "150/170/180/190",
		[20964] = "210/235/250/265",
		[20966] = "160/190/205/220",
		[20967] = "190/220/235/250",
		[20969] = "225/250/265/280",
		[21748] = "200/225/240/255",
		[21752] = "225/235/245/255",
		[21753] = "250/275/290/305",
		[21754] = "245/270/285/300",
		[21755] = "220/245/260/275",
		[21756] = "200/225/240/255",
		[21758] = "215/240/255/270",
		[21760] = "225/250/265/280",
		[21763] = "235/260/275/290",
		[21764] = "235/260/275/290",
		[21765] = "240/265/280/295",
		[21766] = "250/275/290/305",
		[21767] = "260/280/290/300",
		[21768] = "275/285/295/305",
		[21769] = "260/280/290/300",
		[21774] = "275/285/295/305",
		[21775] = "280/290/300/310",
		[21777] = "285/295/305/315",
		[21778] = "285/295/305/315",
		[21779] = "310/320/330/340",
		[21780] = "325/335/345/355",
		[21784] = "300/310/320/330",
		[21789] = "300/310/320/330",
		[21790] = "280/290/300/310",
		[21791] = "290/300/310/320",
		[21792] = "305/315/325/335",
		[21793] = "315/325/335/345",
		[21931] = "1/30/45/60",
		[21932] = "5/35/50/65",
		[21933] = "50/80/95/110",
		[21934] = "30/60/75/90",
		[23094] = "300/300/320/340",
		[23095] = "305/305/325/345",
		[23096] = "315/315/335/355",
		[23097] = "325/325/340/355",
		[23098] = "300/300/320/340",
		[23099] = "305/305/325/345",
		[23100] = "315/315/335/355",
		[23101] = "325/325/340/355",
		[23103] = "300/300/320/340",
		[23104] = "305/305/325/345",
		[23105] = "315/315/335/355",
		[23106] = "325/325/340/355",
		[23108] = "300/300/320/340",
		[23109] = "305/305/325/345",
		[23110] = "315/315/335/355",
		[23111] = "325/325/340/355",
		[23113] = "300/300/320/340",
		[23114] = "305/305/325/345",
		[23115] = "315/315/335/355",
		[23116] = "325/325/340/355",
		[23118] = "300/300/320/340",
		[23119] = "305/305/325/345",
		[23120] = "315/315/335/355",
		[23121] = "325/325/340/355",
		[24027] = "350/350/365/380",
		[24028] = "350/350/365/380",
		[24029] = "350/350/365/380",
		[24030] = "350/350/365/380",
		[24031] = "350/350/365/380",
		[24032] = "350/350/365/380",
		[24033] = "350/350/365/380",
		[24035] = "350/350/365/380",
		[24036] = "350/350/365/380",
		[24037] = "350/350/365/380",
		[24039] = "350/350/365/380",
		[24047] = "350/350/365/380",
		[24048] = "350/350/365/380",
		[24050] = "350/350/365/380",
		[24051] = "350/350/365/380",
		[24052] = "350/350/365/380",
		[24053] = "350/350/365/380",
		[24054] = "350/350/365/380",
		[24055] = "350/350/365/380",
		[24056] = "350/350/365/380",
		[24057] = "350/350/365/380",
		[24058] = "350/350/365/380",
		[24059] = "350/350/365/380",
		[24060] = "350/350/365/380",
		[24061] = "350/350/365/380",
		[24062] = "350/350/365/380",
		[24065] = "350/350/365/380",
		[24066] = "350/350/365/380",
		[24067] = "350/350/365/380",
		[24074] = "310/320/330/340",
		[24075] = "310/320/335/350",
		[24076] = "320/330/340/350",
		[24077] = "335/345/355/365",
		[24078] = "335/345/355/365",
		[24079] = "350/360/370/380",
		[24080] = "355/365/375/385",
		[24082] = "355/365/375/385",
		[24085] = "360/370/380/390",
		[24086] = "365/375/385/395",
		[24087] = "345/355/365/375",
		[24088] = "355/365/375/385",
		[24089] = "365/375/385/395",
		[24092] = "360/370/380/390",
		[24093] = "360/370/380/390",
		[24095] = "360/370/380/390",
		[24097] = "360/370/380/390",
		[24098] = "360/370/380/390",
		[24106] = "355/365/375/385",
		[24110] = "355/365/375/385",
		[24114] = "360/370/380/390",
		[24116] = "360/370/380/390",
		[24117] = "365/375/385/395",
		[24121] = "365/375/385/395",
		[24122] = "370/380/390/400",
		[24123] = "370/380/390/400",
		[24124] = "370/380/390/400",
		[24125] = "370/380/390/400",
		[24126] = "370/380/390/400",
		[24127] = "370/380/390/400",
		[24128] = "370/380/390/400",
		[25438] = "20/50/65/80",
		[25439] = "20/50/65/80",
		[25498] = "1/30/40/50",
		[25880] = "50/70/80/90",
		[25881] = "110/120/130/140",
		[25882] = "175/175/185/195",
		[25883] = "225/225/235/245",
		[25890] = "365/375/385/395",
		[25893] = "365/375/385/395",
		[25894] = "365/375/385/395",
		[25895] = "365/375/385/395",
		[25896] = "365/375/385/395",
		[25897] = "365/375/385/395",
		[25898] = "365/375/385/395",
		[25899] = "365/375/385/395",
		[25901] = "365/375/385/395",
		[28290] = "325/325/340/355",
		[28595] = "305/305/325/345",
		[29157] = "180/190/200/210",
		[29158] = "200/210/220/230",
		[29159] = "280/290/300/310",
		[29160] = "290/300/310/320",
		[30419] = "75/105/120/135",
		[30420] = "105/135/150/165",
		[30421] = "230/255/270/285",
		[30422] = "265/285/295/305",
		[30804] = "65/95/110/125",
		[30825] = "360/370/380/390",
		[31079] = "325/325/335/345",
		[31154] = "80/110/125/140",
		[31398] = "375/385/395/405",
		[31399] = "375/385/395/405",
		[31860] = "325/325/340/355",
		[31861] = "350/350/365/380",
		[31862] = "325/325/340/355",
		[31863] = "350/350/365/380",
		[31864] = "325/325/340/355",
		[31865] = "350/350/365/380",
		[31866] = "325/325/340/355",
		[31867] = "350/350/365/380",
		[31868] = "350/350/365/380",
		[31869] = "325/325/340/355",
		[32193] = "375/375/387/400",
		[32194] = "375/375/387/400",
		[32195] = "375/375/387/400",
		[32196] = "375/375/387/400",
		[32197] = "375/375/387/400",
		[32198] = "375/375/387/400",
		[32199] = "375/375/387/400",
		[32200] = "375/375/387/400",
		[32201] = "375/375/387/400",
		[32202] = "375/375/387/400",
		[32203] = "375/375/387/400",
		[32204] = "375/375/387/400",
		[32205] = "375/375/387/400",
		[32206] = "375/375/387/400",
		[32207] = "375/375/387/400",
		[32208] = "375/375/387/400",
		[32209] = "375/375/387/400",
		[32210] = "375/375/387/400",
		[32211] = "375/375/387/400",
		[32212] = "375/375/387/400",
		[32213] = "375/375/387/400",
		[32214] = "375/375/387/400",
		[32215] = "375/375/387/400",
		[32216] = "375/375/387/400",
		[32217] = "375/375/387/400",
		[32218] = "375/375/387/400",
		[32219] = "375/375/387/400",
		[32220] = "375/375/387/400",
		[32221] = "375/375/387/400",
		[32222] = "375/375/387/400",
		[32223] = "375/375/387/400",
		[32224] = "375/375/387/400",
		[32225] = "375/375/387/400",
		[32226] = "375/375/387/400",
		[32409] = "365/375/385/395",
		[32410] = "365/375/385/395",
		[32508] = "340/340/355/370",
		[32772] = "325/335/345/355",
		[32774] = "330/340/350/360",
		[32776] = "375/375/385/395",
		[32833] = "325/325/332/340",
		[32836] = "350/350/365/380",
		[33131] = "360/365/370/375",
		[33133] = "360/365/370/375",
		[33134] = "360/365/370/375",
		[33135] = "360/365/370/375",
		[33140] = "360/365/370/375",
		[33143] = "360/365/370/375",
		[33144] = "360/365/370/375",
		[33782] = "350/350/365/380",
		[34220] = "365/375/385/395",
		[34358] = "365/375/392/410",
		[34359] = "365/375/392/410",
		[34360] = "365/375/392/410",
		[34361] = "365/375/392/410",
		[34362] = "365/375/392/410",
		[34363] = "365/375/392/410",
		[35315] = "350/350/365/380",
		[35316] = "350/350/365/380",
		[35318] = "350/350/365/380",
		[35501] = "370/380/390/400",
		[35503] = "370/380/390/400",
		[35693] = "375/375/387/400",
		[35694] = "375/375/387/400",
		[35700] = "375/375/387/400",
		[35702] = "375/375/387/400",
		[35703] = "375/375/387/400",
		[35707] = "350/350/365/380",
		[35758] = "375/375/387/400",
		[35759] = "375/375/387/400",
		[35760] = "375/375/387/400",
		[35761] = "375/375/387/400",
		[35945] = "350/350/365/380",
		[37503] = "375/375/387/400",
		
		-- leatherworking
		[2300] = "40/70/85/100",
		[2302] = "1/40/55/70",
		[2303] = "15/45/60/75",
		[2304] = "1/30/45/60",
		[2307] = "90/120/135/150",
		[2308] = "85/105/120/135",
		[2309] = "55/85/100/115",
		[2310] = "60/90/105/120",
		[2311] = "60/90/105/120",
		[2312] = "75/105/120/135",
		[2313] = "100/115/122/130",
		[2314] = "120/145/157/170",
		[2315] = "100/125/137/150",
		[2316] = "110/135/147/160",
		[2317] = "100/125/137/150",
		[2318] = "1/20/30/40",
		[2319] = "100/100/105/110",
		[3719] = "150/170/180/190",
		[4231] = "35/55/65/75",
		[4233] = "100/115/122/130",
		[4234] = "150/150/155/160",
		[4236] = "150/160/165/170",
		[4237] = "25/55/70/85",
		[4239] = "55/85/100/115",
		[4242] = "75/105/120/135",
		[4243] = "85/115/130/145",
		[4244] = "100/125/137/150",
		[4246] = "80/110/125/140",
		[4247] = "145/170/182/195",
		[4248] = "120/155/167/180",
		[4249] = "125/150/162/175",
		[4250] = "120/145/157/170",
		[4251] = "130/155/167/180",
		[4252] = "140/165/177/190",
		[4253] = "135/160/172/185",
		[4254] = "150/170/180/190",
		[4255] = "155/175/185/195",
		[4256] = "175/195/205/215",
		[4257] = "160/180/190/200",
		[4258] = "170/190/200/210",
		[4259] = "180/200/210/220",
		[4260] = "195/215/225/235",
		[4262] = "185/205/215/225",
		[4264] = "200/220/230/240",
		[4265] = "150/170/180/190",
		[4304] = "200/200/202/205",
		[4455] = "165/185/195/205",
		[4456] = "165/185/195/205",
		[5081] = "40/70/85/100",
		[5739] = "190/210/220/230",
		[5780] = "90/120/135/150",
		[5781] = "95/125/140/155",
		[5782] = "170/190/200/210",
		[5783] = "190/210/220/230",
		[5957] = "1/40/55/70",
		[5958] = "105/130/142/155",
		[5961] = "115/140/152/165",
		[5962] = "160/180/190/200",
		[5963] = "170/190/200/210",
		[5964] = "175/195/205/215",
		[5965] = "185/205/215/225",
		[5966] = "190/210/220/230",
		[6466] = "90/120/135/150",
		[6467] = "105/130/142/155",
		[6468] = "115/140/152/165",
		[6709] = "90/115/130/145",
		[7276] = "1/40/55/70",
		[7277] = "1/40/55/70",
		[7278] = "30/60/75/90",
		[7279] = "30/60/75/90",
		[7280] = "35/65/80/95",
		[7281] = "70/100/115/130",
		[7282] = "95/125/140/155",
		[7283] = "100/125/137/150",
		[7284] = "120/145/157/170",
		[7285] = "120/145/157/170",
		[7348] = "125/150/162/175",
		[7349] = "135/160/172/185",
		[7352] = "135/160/172/185",
		[7358] = "140/165/177/190",
		[7359] = "145/170/182/195",
		[7371] = "150/170/180/190",
		[7372] = "150/170/180/190",
		[7373] = "165/185/195/205",
		[7374] = "175/195/205/215",
		[7375] = "175/195/205/215",
		[7377] = "180/200/210/220",
		[7378] = "185/205/215/225",
		[7386] = "190/210/220/230",
		[7387] = "195/215/225/235",
		[7390] = "200/220/230/240",
		[7391] = "200/220/230/240",
		[8170] = "250/250/250/250",
		[8172] = "200/200/200/200",
		[8173] = "200/220/230/240",
		[8174] = "200/220/230/240",
		[8175] = "205/225/235/245",
		[8176] = "205/225/235/245",
		[8185] = "235/255/265/275",
		[8187] = "205/225/235/245",
		[8189] = "210/230/240/250",
		[8191] = "230/250/260/270",
		[8192] = "210/230/240/250",
		[8193] = "230/250/260/270",
		[8197] = "235/255/265/275",
		[8198] = "210/230/240/250",
		[8200] = "215/235/245/255",
		[8201] = "220/240/250/260",
		[8202] = "240/260/270/280",
		[8203] = "220/240/250/260",
		[8204] = "225/245/255/265",
		[8205] = "220/240/250/260",
		[8206] = "245/265/275/285",
		[8207] = "240/260/270/280",
		[8208] = "250/270/280/290",
		[8209] = "235/255/265/275",
		[8210] = "220/240/250/260",
		[8211] = "225/245/255/265",
		[8212] = "250/270/280/290",
		[8213] = "245/265/275/285",
		[8214] = "225/245/255/265",
		[8215] = "250/270/280/290",
		[8216] = "240/260/270/280",
		[8217] = "225/245/255/265",
		[8218] = "225/245/255/265",
		[8345] = "225/245/255/265",
		[8346] = "230/250/260/270",
		[8347] = "225/245/255/265",
		[8348] = "250/270/280/290",
		[8349] = "250/270/280/290",
		[8367] = "255/275/285/295",
		[15045] = "260/280/290/300",
		[15046] = "270/290/300/310",
		[15047] = "300/320/330/340",
		[15048] = "285/305/315/325",
		[15049] = "295/315/325/335",
		[15050] = "290/310/320/330",
		[15051] = "300/320/330/340",
		[15052] = "300/320/330/340",
		[15053] = "285/305/315/325",
		[15054] = "270/290/300/310",
		[15055] = "300/320/330/340",
		[15056] = "285/305/315/325",
		[15057] = "275/295/305/315",
		[15058] = "295/315/325/335",
		[15059] = "300/320/330/340",
		[15060] = "285/305/315/325",
		[15061] = "270/290/300/310",
		[15062] = "300/320/330/340",
		[15063] = "290/310/320/330",
		[15064] = "275/295/305/315",
		[15065] = "285/305/315/325",
		[15066] = "290/310/320/330",
		[15067] = "270/290/300/310",
		[15068] = "300/320/330/340",
		[15069] = "285/305/315/325",
		[15070] = "295/315/325/335",
		[15071] = "275/295/305/315",
		[15072] = "280/300/310/320",
		[15073] = "275/295/305/315",
		[15074] = "265/285/295/305",
		[15075] = "290/310/320/330",
		[15076] = "265/285/295/305",
		[15077] = "255/275/285/295",
		[15078] = "275/295/305/315",
		[15079] = "285/305/315/325",
		[15080] = "295/315/325/335",
		[15081] = "300/320/330/340",
		[15082] = "280/300/310/320",
		[15083] = "260/280/290/300",
		[15084] = "265/285/295/305",
		[15085] = "300/320/330/340",
		[15086] = "280/300/310/320",
		[15087] = "290/310/320/330",
		[15088] = "300/320/330/340",
		[15090] = "300/320/330/340",
		[15091] = "270/290/300/310",
		[15092] = "275/295/305/315",
		[15093] = "280/300/310/320",
		[15094] = "290/310/320/330",
		[15095] = "300/320/330/340",
		[15096] = "300/320/330/340",
		[15138] = "300/320/330/340",
		[15407] = "250/250/255/260",
		[15564] = "250/250/260/270",
		[16982] = "295/315/325/335",
		[16983] = "300/320/330/340",
		[16984] = "300/320/330/340",
		[17721] = "190/210/220/230",
		[18238] = "200/210/220/230",
		[18251] = "300/320/330/340",
		[18258] = "285/285/290/295",
		[18504] = "300/320/330/340",
		[18506] = "300/320/330/340",
		[18508] = "300/320/330/340",
		[18509] = "300/320/330/340",
		[18510] = "300/320/330/340",
		[18511] = "300/320/330/340",
		[18662] = "150/150/155/160",
		[18948] = "155/175/185/195",
		[19044] = "290/310/320/330",
		[19049] = "300/320/330/340",
		[19052] = "290/310/320/330",
		[19058] = "300/320/330/340",
		[19149] = "300/320/330/340",
		[19157] = "300/320/330/340",
		[19162] = "300/320/330/340",
		[19163] = "300/320/330/340",
		[19685] = "300/320/330/340",
		[19686] = "300/320/330/340",
		[19687] = "300/320/330/340",
		[19688] = "300/320/330/340",
		[19689] = "300/320/330/340",
		[20295] = "300/320/330/340",
		[20296] = "280/300/310/320",
		[20380] = "300/320/330/340",
		[20476] = "300/320/330/340",
		[20477] = "300/320/330/340",
		[20478] = "300/320/330/340",
		[20479] = "300/320/330/340",
		[20480] = "300/320/330/340",
		[20481] = "300/320/330/340",
		[20575] = "100/125/137/150",
		[21278] = "300/320/330/340",
		[21887] = "300/300/305/310",
		[22661] = "300/320/330/340",
		[22662] = "300/320/330/340",
		[22663] = "300/320/330/340",
		[22664] = "300/320/330/340",
		[22665] = "300/320/330/340",
		[22666] = "300/320/330/340",
		[22759] = "300/320/330/340",
		[22760] = "300/320/330/340",
		[22761] = "300/320/330/340",
		[23793] = "325/325/330/335",
		[25650] = "300/310/325/340",
		[25651] = "325/335/340/345",
		[25652] = "325/335/340/345",
		[25653] = "350/360/370/380",
		[25654] = "300/310/320/330",
		[25655] = "310/320/330/340",
		[25656] = "320/330/340/350",
		[25657] = "335/345/355/365",
		[25659] = "335/345/355/365",
		[25660] = "325/335/345/355",
		[25661] = "310/320/330/340",
		[25662] = "300/310/320/330",
		[25668] = "320/330/340/350",
		[25669] = "300/310/320/330",
		[25670] = "315/325/335/345",
		[25671] = "330/340/350/360",
		[25673] = "300/310/320/330",
		[25674] = "310/320/330/340",
		[25675] = "320/330/340/350",
		[25676] = "330/340/350/360",
		[25679] = "300/300/305/310",
		[25680] = "350/360/370/380",
		[25681] = "350/360/370/380",
		[25682] = "350/360/370/380",
		[25683] = "350/360/370/380",
		[25685] = "340/350/360/370",
		[25686] = "350/360/370/380",
		[25687] = "350/360/370/380",
		[25689] = "360/370/380/390",
		[25690] = "355/365/375/385",
		[25691] = "355/365/375/385",
		[25692] = "340/350/360/370",
		[25693] = "350/360/370/380",
		[25694] = "340/350/360/370",
		[25695] = "350/360/370/380",
		[25696] = "360/370/380/390",
		[25697] = "360/370/380/390",
		[29483] = "340/350/355/360",
		[29485] = "340/350/355/360",
		[29486] = "340/350/355/360",
		[29487] = "340/350/355/360",
		[29488] = "340/350/355/360",
		[29489] = "350/360/370/380",
		[29490] = "350/360/370/380",
		[29491] = "350/360/370/380",
		[29492] = "350/360/370/380",
		[29493] = "350/360/370/380",
		[29494] = "350/360/370/380",
		[29495] = "350/360/370/380",
		[29496] = "350/360/370/380",
		[29497] = "350/360/370/380",
		[29498] = "350/360/370/380",
		[29499] = "350/360/370/380",
		[29500] = "350/360/370/380",
		[29502] = "365/375/385/395",
		[29503] = "365/375/385/395",
		[29504] = "365/375/385/395",
		[29505] = "365/375/385/395",
		[29506] = "365/375/385/395",
		[29507] = "365/375/385/395",
		[29508] = "365/375/385/395",
		[29509] = "365/375/385/395",
		[29510] = "365/375/385/395",
		[29511] = "365/375/385/395",
		[29512] = "365/375/385/395",
		[29514] = "365/375/385/395",
		[29515] = "375/385/395/405",
		[29516] = "375/385/395/405",
		[29517] = "375/385/395/405",
		[29519] = "375/385/395/405",
		[29520] = "375/385/395/405",
		[29521] = "375/385/395/405",
		[29522] = "375/385/395/405",
		[29523] = "375/385/395/405",
		[29524] = "375/385/395/405",
		[29525] = "375/385/395/405",
		[29526] = "375/385/395/405",
		[29527] = "375/385/395/405",
		[29528] = "340/340/347/355",
		[29529] = "365/365/372/380",
		[29530] = "345/345/352/360",
		[29531] = {[35539] = "350/350/357/365", [351769] = "350/350/357/365"},
		[29532] = "370/370/377/385",
		[29533] = "335/335/345/355",
		[29534] = "335/335/345/355",
		[29535] = "365/365/375/385",
		[29536] = "365/365/375/385",
		[29540] = "325/335/340/345",
		[29964] = "260/280/290/300",
		[29970] = "260/280/290/300",
		[29971] = "260/280/290/300",
		[29973] = "330/350/360/370",
		[29974] = "330/350/360/370",
		[29975] = "330/350/360/370",
		[30039] = "375/385/395/405",
		[30040] = "375/385/395/405",
		[30041] = "375/385/395/405",
		[30042] = "375/385/395/405",
		[30043] = "375/385/395/405",
		[30044] = "375/385/395/405",
		[30045] = "375/385/395/405",
		[30046] = "375/385/395/405",
		[32393] = "375/385/395/405",
		[32394] = "375/385/395/405",
		[32395] = "375/385/395/405",
		[32396] = "375/385/395/405",
		[32397] = "375/385/395/405",
		[32398] = "375/385/395/405",
		[32399] = "375/385/395/405",
		[32400] = "375/385/395/405",
		[32574] = "375/385/395/405",
		[32575] = "375/385/395/405",
		[32577] = "375/385/395/405",
		[32579] = "375/385/395/405",
		[32580] = "375/385/395/405",
		[32581] = "375/385/395/405",
		[32582] = "375/385/395/405",
		[32583] = "375/385/395/405",
		[33122] = "360/370/380/390",
		[33204] = "365/375/385/395",
		[34086] = "285/285/285/285",
		[34099] = "325/325/335/345",
		[34100] = "325/325/335/345",
		[34105] = "350/360/370/380",
		[34106] = "350/360/370/380",
		[34207] = "350/355/360/365",
		[34330] = "350/355/360/365",
		[34369] = "365/375/392/410",
		[34370] = "365/375/392/410",
		[34371] = "365/375/392/410",
		[34372] = "365/375/392/410",
		[34373] = "365/375/392/410",
		[34374] = "365/375/392/410",
		[34375] = "365/375/392/410",
		[34376] = "365/375/392/410",
		[34482] = "300/310/320/330",
		[34490] = "360/370/380/390",
		[185848] = "365/365/372/380",
		[185849] = "370/370/377/385",
		[185851] = "345/345/352/360",
		[185852] = "375/375/375/375",
		
		-- mining
		[2840] = "1/25/47/70",
		[2841] = "65/65/90/115",
		[2842] = "75/115/122/130",
		[3575] = "125/130/145/160",
		[3576] = "65/65/70/75",
		[3577] = "155/170/177/185",
		[3859] = "165/165/165/165",
		[3860] = "175/175/202/230",
		[6037] = "230/235/242/250",
		[11371] = "300/300/305/310",
		[12359] = "250/250/270/290",
		[17771] = "350/350/362/375",
		[22573] = "300/300/300/300",
		[22574] = "300/300/300/300",
		[23445] = "300/300/307/315",
		[23446] = "325/325/332/340",
		[23447] = "350/350/357/365",
		[23448] = "350/355/367/380",
		[23449] = "375/375/375/375",
		[23573] = "375/375/375/375",
		[35128] = "375/375/375/375",
		
		-- tailoring
		[2568] = "10/45/57/70",
		[2569] = "65/90/107/125",
		[2570] = "1/35/47/60",
		[2572] = "40/65/82/100",
		[2575] = "40/65/82/100",
		[2576] = "1/35/47/60",
		[2577] = "40/65/82/100",
		[2578] = "70/95/112/130",
		[2579] = "70/95/112/130",
		[2580] = "60/85/102/120",
		[2582] = "85/110/127/145",
		[2583] = "95/120/137/155",
		[2584] = "75/100/117/135",
		[2585] = "105/130/147/165",
		[2587] = "100/110/120/130",
		[2996] = "1/25/37/50",
		[2997] = "75/90/97/105",
		[4238] = "45/70/87/105",
		[4240] = "80/105/122/140",
		[4241] = "95/120/137/155",
		[4245] = "150/170/185/200",
		[4305] = "125/135/140/145",
		[4307] = "35/60/77/95",
		[4308] = "60/85/102/120",
		[4309] = "70/95/112/130",
		[4310] = "85/110/127/145",
		[4311] = "100/125/142/160",
		[4312] = "80/105/122/140",
		[4313] = "95/120/137/155",
		[4314] = "110/135/152/170",
		[4315] = "120/145/162/180",
		[4316] = "110/135/152/170",
		[4317] = "125/150/167/185",
		[4318] = "130/150/165/180",
		[4319] = "145/165/180/195",
		[4320] = "125/150/167/185",
		[4321] = "140/160/175/190",
		[4322] = "165/185/200/215",
		[4323] = "170/190/205/220",
		[4324] = "150/170/185/200",
		[4325] = "175/195/210/225",
		[4326] = "185/205/220/235",
		[4327] = "200/220/235/250",
		[4328] = "180/200/215/230",
		[4329] = "200/220/235/250",
		[4330] = "110/135/152/170",
		[4331] = "125/150/167/185",
		[4332] = "135/145/150/155",
		[4333] = "155/165/170/175",
		[4334] = "170/180/185/190",
		[4335] = "185/195/200/205",
		[4336] = "200/210/215/220",
		[4339] = "175/180/182/185",
		[4343] = "30/55/72/90",
		[4344] = "1/35/47/60",
		[5542] = "90/115/132/150",
		[5762] = "70/95/112/130",
		[5763] = "115/140/157/175",
		[5764] = "175/195/210/225",
		[5765] = "185/205/220/235",
		[5766] = "135/155/170/185",
		[5770] = "150/170/185/200",
		[6238] = "30/55/72/90",
		[6239] = "55/80/97/115",
		[6240] = "55/80/97/115",
		[6241] = "30/55/72/90",
		[6242] = "70/95/112/130",
		[6263] = "100/125/142/160",
		[6264] = "115/140/157/175",
		[6384] = "120/145/162/180",
		[6385] = "120/145/162/180",
		[6786] = "40/65/82/100",
		[6787] = "110/135/152/170",
		[6795] = "160/170/175/180",
		[6796] = "175/185/190/195",
		[6836] = "250/255/270/285",
		[7026] = "15/50/67/85",
		[7046] = "140/160/175/190",
		[7047] = "145/165/180/195",
		[7048] = "145/155/160/165",
		[7049] = "150/170/185/200",
		[7050] = "160/170/175/180",
		[7051] = "170/190/205/220",
		[7052] = "175/195/210/225",
		[7053] = "175/195/210/225",
		[7054] = "190/210/225/240",
		[7055] = "175/195/210/225",
		[7056] = "180/200/215/230",
		[7057] = "180/200/215/230",
		[7058] = "185/205/215/225",
		[7059] = "190/210/225/240",
		[7060] = "190/210/225/240",
		[7061] = "195/215/230/245",
		[7062] = "195/215/225/235",
		[7063] = "205/220/235/250",
		[7064] = "210/225/240/255",
		[7065] = "165/185/200/215",
		[9998] = "205/220/235/250",
		[9999] = "205/220/235/250",
		[10001] = "210/225/240/255",
		[10002] = "210/225/240/255",
		[10003] = "215/230/245/260",
		[10004] = "215/230/245/260",
		[10007] = "215/230/245/260",
		[10008] = "215/220/225/230",
		[10009] = "215/230/245/260",
		[10018] = "225/240/255/270",
		[10019] = "225/240/255/270",
		[10021] = "225/240/255/270",
		[10023] = "225/240/255/270",
		[10024] = "230/245/260/275",
		[10025] = "245/260/275/290",
		[10026] = "230/245/260/275",
		[10027] = "230/245/260/275",
		[10028] = "235/250/265/280",
		[10029] = "235/250/265/280",
		[10030] = "240/255/270/285",
		[10031] = "240/255/270/285",
		[10033] = "240/255/270/285",
		[10034] = "240/245/250/255",
		[10035] = "245/250/255/260",
		[10036] = "250/265/280/295",
		[10040] = "250/255/260/265",
		[10041] = "250/265/280/295",
		[10042] = "225/240/255/270",
		[10044] = "245/260/275/290",
		[10045] = "1/35/47/60",
		[10046] = "20/50/67/85",
		[10047] = "75/100/117/135",
		[10048] = "120/145/162/180",
		[10050] = "225/240/255/270",
		[10051] = "235/250/265/280",
		[10052] = "220/225/230/235",
		[10053] = "235/240/245/250",
		[10054] = "230/235/240/245",
		[10055] = "235/240/245/250",
		[10056] = "215/220/225/230",
		[13856] = "255/270/285/300",
		[13857] = "260/275/290/305",
		[13858] = "260/275/290/305",
		[13860] = "265/280/295/310",
		[13863] = "275/290/305/320",
		[13864] = "280/295/310/325",
		[13865] = "285/300/315/330",
		[13866] = "295/310/325/340",
		[13867] = "300/315/330/345",
		[13868] = "255/270/285/300",
		[13869] = "255/270/285/300",
		[13870] = "265/280/295/310",
		[13871] = "280/295/310/325",
		[14042] = "260/275/290/305",
		[14043] = "270/285/300/315",
		[14044] = "275/290/305/320",
		[14045] = "280/295/310/325",
		[14046] = "260/275/290/305",
		[14048] = "250/255/257/260",
		[14100] = "270/285/300/315",
		[14101] = "270/285/300/315",
		[14103] = "275/290/305/320",
		[14104] = "290/305/320/335",
		[14106] = "300/315/330/345",
		[14107] = "275/290/305/320",
		[14108] = "285/300/315/330",
		[14111] = "290/305/320/335",
		[14112] = "300/315/330/345",
		[14128] = "300/315/330/345",
		[14130] = "300/315/330/345",
		[14132] = "275/290/305/320",
		[14134] = "275/290/305/320",
		[14136] = "285/300/315/330",
		[14137] = "290/305/320/335",
		[14138] = "300/315/330/345",
		[14139] = "300/315/330/345",
		[14140] = "300/315/330/345",
		[14141] = "275/290/305/320",
		[14142] = "270/285/300/315",
		[14143] = "265/280/295/310",
		[14144] = "290/305/320/335",
		[14146] = "300/315/330/345",
		[14152] = "300/315/330/345",
		[14153] = "300/315/330/345",
		[14154] = "300/315/330/345",
		[14155] = "300/315/330/345",
		[14156] = "300/315/330/345",
		[14342] = "250/290/305/320",
		[15802] = "290/295/310/325",
		[16979] = "300/315/330/345",
		[16980] = "300/315/330/345",
		[17723] = "190/200/205/210",
		[18258] = "285/285/290/295",
		[18263] = "300/320/335/350",
		[18405] = "300/315/330/345",
		[18407] = "300/315/330/345",
		[18408] = "300/315/330/345",
		[18409] = "300/315/330/345",
		[18413] = "300/315/330/345",
		[18486] = "300/315/330/345",
		[19047] = "290/305/320/335",
		[19050] = "300/315/330/345",
		[19056] = "290/305/320/335",
		[19059] = "300/315/330/345",
		[19156] = "300/315/330/345",
		[19165] = "300/315/330/345",
		[19682] = "300/315/330/345",
		[19683] = "300/315/330/345",
		[19684] = "300/315/330/345",
		[20537] = "300/315/330/345",
		[20538] = "300/315/330/345",
		[20539] = "300/315/330/345",
		[21154] = "250/265/280/295",
		[21340] = "260/275/290/305",
		[21341] = "285/300/315/330",
		[21342] = "300/315/330/345",
		[21542] = "250/265/280/295",
		[21840] = "300/305/315/325",
		[21841] = "315/320/330/340",
		[21842] = "325/330/335/340",
		[21843] = "340/340/345/350",
		[21844] = "345/345/350/355",
		[21845] = "350/350/355/360",
		[21846] = "355/365/370/375",
		[21847] = "365/375/380/385",
		[21848] = "375/385/390/395",
		[21849] = "310/320/325/330",
		[21850] = "310/320/325/330",
		[21851] = "320/330/335/340",
		[21852] = "325/335/340/345",
		[21853] = "335/345/350/355",
		[21854] = "340/350/355/360",
		[21855] = "345/355/360/365",
		[21858] = "375/385/390/395",
		[21859] = "340/350/355/360",
		[21860] = "350/360/365/370",
		[21861] = "360/370/375/380",
		[21862] = "360/370/375/380",
		[21863] = "355/365/370/375",
		[21864] = "365/375/380/385",
		[21865] = "375/385/390/395",
		[21866] = "350/360/365/370",
		[21867] = "360/370/375/380",
		[21868] = "370/380/385/390",
		[21869] = "355/365/370/375",
		[21870] = "365/375/380/385",
		[21871] = "375/385/390/395",
		[21872] = "375/385/390/395",
		[21873] = "355/365/370/375",
		[21874] = "365/375/380/385",
		[21875] = "375/385/390/395",
		[21876] = "375/385/390/395",
		[22246] = "225/240/255/270",
		[22248] = "275/290/305/320",
		[22249] = "300/315/330/345",
		[22251] = "275/290/305/320",
		[22252] = "300/315/330/345",
		[22652] = "300/315/330/345",
		[22654] = "300/315/330/345",
		[22655] = "300/315/330/345",
		[22658] = "300/315/330/345",
		[22660] = "300/315/330/345",
		[22756] = "300/315/330/345",
		[22757] = "300/315/330/345",
		[22758] = "300/315/330/345",
		[24249] = "350/360/365/370",
		[24250] = "350/360/365/370",
		[24251] = "350/360/365/370",
		[24252] = "350/360/365/370",
		[24253] = "350/360/365/370",
		[24254] = "350/360/365/370",
		[24255] = "365/375/380/385",
		[24256] = "365/375/380/385",
		[24257] = "365/375/380/385",
		[24258] = "365/375/380/385",
		[24259] = "365/375/380/385",
		[24260] = "365/375/380/385",
		[24261] = "375/385/390/395",
		[24262] = "375/385/390/395",
		[24263] = "375/385/390/395",
		[24264] = "375/385/390/395",
		[24266] = "375/385/390/395",
		[24267] = "375/385/390/395",
		[24268] = "300/300/310/320",
		[24270] = "340/350/355/360",
		[24271] = "350/350/355/360",
		[24272] = "350/350/355/360",
		[24273] = "335/345/350/355",
		[24274] = "375/385/390/395",
		[24275] = "335/345/350/355",
		[24276] = "375/385/390/395",
		[30035] = "375/385/390/395",
		[30036] = "375/385/390/395",
		[30037] = "375/385/390/395",
		[30038] = "375/385/390/395",
		[30831] = "350/360/365/370",
		[30837] = "350/360/365/370",
		[30838] = "360/370/375/380",
		[30839] = "370/380/385/390",
		[32389] = "375/385/390/395",
		[32390] = "375/385/390/395",
		[32391] = "375/385/390/395",
		[32392] = "375/385/390/395",
		[32420] = "375/385/390/395",
		[32584] = "375/385/390/395",
		[32585] = "375/385/390/395",
		[32586] = "375/385/390/395",
		[32587] = "375/385/390/395",
		[34085] = "250/250/250/250",
		[34087] = "250/250/250/250",
		[34364] = "365/375/392/410",
		[34365] = "365/375/392/410",
		[34366] = "365/375/392/410",
		[34367] = "365/375/392/410",
		[38225] = "375/385/390/395",
		[38277] = "250/265/280/295",
		[38278] = "245/250/255/260",
		
		-- cooking
		[724] = "50/90/110/130",
		[733] = "75/115/135/155",
		[787] = "1/45/65/85",
		[1017] = "100/140/160/180",
		[1082] = "100/135/155/175",
		[2679] = "1/45/65/85",
		[2680] = "10/50/70/90",
		[2681] = "1/45/65/85",
		[2682] = "85/125/145/165",
		[2683] = "75/115/135/155",
		[2684] = "50/90/110/130",
		[2685] = "110/130/150/170",
		[2687] = "80/120/140/160",
		[2888] = "25/60/80/100",
		[3220] = "60/100/120/140",
		[3662] = "80/120/140/160",
		[3663] = "90/130/150/170",
		[3664] = "120/160/180/200",
		[3665] = "130/170/190/210",
		[3666] = "110/150/170/190",
		[3726] = "110/150/170/190",
		[3727] = "125/175/195/215",
		[3728] = "150/190/210/230",
		[3729] = "175/215/235/255",
		[4457] = "175/215/235/255",
		[4592] = "50/90/110/130",
		[4593] = "100/140/160/180",
		[4594] = "175/190/210/230",
		[5095] = "50/90/110/130",
		[5472] = "10/50/70/90",
		[5473] = "20/60/80/100",
		[5474] = "35/75/95/115",
		[5476] = "50/90/110/130",
		[5477] = "50/90/110/130",
		[5478] = "90/130/150/170",
		[5479] = "100/140/160/180",
		[5480] = "110/150/170/190",
		[5525] = "50/90/110/130",
		[5526] = "90/130/150/170",
		[5527] = "125/165/185/205",
		[6038] = "175/215/235/255",
		[6290] = "1/45/65/85",
		[6316] = "50/90/110/130",
		[6657] = "85/125/145/165",
		[6887] = "225/250/262/275",
		[6888] = "1/45/65/85",
		[6890] = "40/80/100/120",
		[7676] = "60/100/120/140",
		[8364] = "175/215/235/255",
		[10841] = "175/175/190/205",
		[12209] = "125/165/185/205",
		[12210] = "175/215/235/255",
		[12212] = "175/215/235/255",
		[12213] = "175/215/235/255",
		[12214] = "175/215/235/255",
		[12215] = "200/225/237/250",
		[12216] = "225/250/262/275",
		[12217] = "200/225/237/250",
		[12218] = "225/250/262/275",
		[12224] = "1/45/65/85",
		[13851] = "175/215/235/255",
		[13927] = "225/250/262/275",
		[13928] = "240/265/277/290",
		[13929] = "240/265/277/290",
		[13930] = "225/250/262/275",
		[13931] = "250/275/285/295",
		[13932] = "250/275/285/295",
		[13933] = "275/300/312/325",
		[13934] = "275/300/312/325",
		[13935] = "275/300/312/325",
		[16766] = "225/250/262/275",
		[17197] = "1/45/65/85",
		[17198] = "35/75/95/115",
		[17222] = "200/225/237/250",
		[18045] = "225/250/262/275",
		[18254] = "275/300/312/325",
		[20074] = "150/160/180/200",
		[20452] = "285/310/322/335",
		[21023] = "300/325/337/350",
		[21072] = "80/120/140/160",
		[21217] = "175/215/235/255",
		[22645] = "60/100/120/140",
		[24105] = "1/45/65/85",
		[27635] = "1/45/65/85",
		[27636] = "50/90/110/130",
		[27651] = "300/320/330/340",
		[27655] = "300/320/330/340",
		[27656] = "310/330/340/350",
		[27657] = "315/335/345/355",
		[27658] = "325/345/355/365",
		[27659] = "325/345/355/365",
		[27660] = "325/345/355/365",
		[27661] = "300/320/330/340",
		[27662] = "300/320/330/340",
		[27663] = "310/330/340/350",
		[27664] = "320/340/350/360",
		[27665] = "320/340/350/360",
		[27666] = "325/345/355/365",
		[27667] = "350/370/380/390",
		[30155] = "300/320/330/340",
		[30816] = "1/30/35/40",
		[31672] = "335/355/365/375",
		[31673] = "335/355/365/375",
		[33048] = "335/335/345/355",
		[33052] = "375/375/380/385",
		[33053] = "375/375/380/385",
		[33825] = "325/335/345/355",
		[33866] = "300/320/330/340",
		[33867] = "300/320/330/340",
		[33872] = "325/335/345/355",
		[33874] = "300/345/355/365",
		[33924] = "1/50/62/75",
		[34411] = "325/325/325/325",
		[34832] = "100/100/105/110",
		[35563] = "250/275/285/295",
		[35565] = "250/275/285/295",
		
		-- first-aid
		[1251] = "1/30/45/60",
		[2581] = "40/50/75/100",
		[3530] = "80/80/115/150",
		[3531] = "115/115/150/185",
		[6450] = "150/150/180/210",
		[6451] = "180/180/210/240",
		[6452] = "80/80/115/150",
		[6453] = "130/130/165/200",
		[8544] = "210/210/240/270",
		[8545] = "240/240/270/300",
		[14529] = "260/260/290/320",
		[14530] = "290/290/320/350",
		[19440] = "300/300/330/360",
		[21990] = "330/330/360/390",
		[21991] = "360/360/385/410",
	}
end
